
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_outline.c */

#include "fs_itype.h"
#include "fs_bitmap.h"
#include "fs_fontscal.h"
#ifdef FS_EDGE_TECH
#include "adf.h"
#endif

#ifdef FS_RENDER

/* local prototypes */
FS_OUTLINE *extract_outline(_DS_ fsg_SplineKey *key);
FS_VOID fill_outline(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_LONG *px, FS_LONG *py, FS_BYTE *pf, FS_OUTLINE *outl);
#ifdef FS_CFFR
FS_VOID fill_outline_cff(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_LONG *px, FS_LONG *py, FS_BYTE *pf, FS_OUTLINE *outl);
static FS_VOID count_points_cff(FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_BYTE *pf, FS_LONG *_types, FS_LONG *_points);
#endif
FS_VOID count_points(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_BYTE *pf, FS_LONG *_types, FS_LONG *_points);

/****************************************************************/
static FS_VOID set_bbox(int n_points, FS_OUTLINE *outl)
{
    FS_FIXED lo_x, hi_x, lo_y, hi_y, x, y;
    int i;

    if (n_points == 0)
        outl->lo_x = outl->hi_x = outl->lo_y = outl->hi_y = 0;
    else
    {
        FS_FIXED *px, *py;
        px = (FS_FIXED *)(outl->x);
        lo_x = hi_x = px[0];
        py = (FS_FIXED *)(outl->y);
        lo_y = hi_y = py[0];
        for (i = 1; i < n_points; i++)
        {
            x = px[i];
            if (x > hi_x) hi_x = x;
            if (x < lo_x) lo_x = x;
            y = py[i];
            if (y > hi_y) hi_y = y;
            if (y < lo_y) lo_y = y;
        }
        outl->lo_x = lo_x;
        outl->lo_y = lo_y;
        outl->hi_x = hi_x;
        outl->hi_y = hi_y;
    }
}

/****************************************************************/
FS_OUTLINE *extract_outline(_DS_ fsg_SplineKey *key)
{
    fnt_ElementType *elementPtr;
    FS_BYTE *workSpacePtr;
    fsg_OffsetInfo *offsetPtr;
    FS_OUTLINE *outl;

    FS_LONG *px, *py;
    FS_BYTE *pf;
    FS_SHORT *ps, *pe;
    FS_SHORT nc, np, min;
    FS_LONG n_types = 0, n_points = 0;
    TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;

#ifdef FS_PSEUDO_BOLD
    int is_pseudo;
#endif

#if defined(FS_PSEUDO_BOLD) || defined(FS_STIK)
    int is_stik = 0;
#endif

    /* shouldn't all the values in elementPtr be valid ? */
    elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
    workSpacePtr = (FS_BYTE *)(STATE.server->workspace);
    offsetPtr = &(key->elementInfoRec.offsets[GLYPHELEMENT]);

    /* scaled x and y coords and flags*/
    px = (FS_LONG *)(workSpacePtr + offsetPtr->newXOffset);
    py = (FS_LONG *)(workSpacePtr + offsetPtr->newYOffset);
    pf = (FS_BYTE *)(workSpacePtr + offsetPtr->onCurveOffset);

    /* contour start, end, and count */
    ps = (FS_SHORT *)(workSpacePtr + offsetPtr->startPointOffset);
    pe = (FS_SHORT *)(workSpacePtr + offsetPtr->endPointOffset);
    nc = elementPtr->nc;

    /* number of ON/OFF curve points */
    np = nc ? 1 + pe[nc - 1] : 0;


#if defined(FS_STIK) && defined(FS_ACT3)
    /* restore the "I'm really an outline" bit flag in compressed STIK fonts */
    if (np && (STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&
            (STATE.cur_lfnt->fontflags & FONTFLAG_ACT3))
        pf[0] |= STATE.outl_char;
    else    /* possible a CCC font */
    {
        if (np && STIK_FORMAT_CCC(ttf->head->glyphDataFormat) )
            pf[0] |= STATE.outl_char;
    }
#else
#if defined(FS_STIK) && defined(FS_CCC)

    if (np && STIK_FORMAT_CCC(ttf->head->glyphDataFormat) )
        pf[0] |= STATE.outl_char;
#endif
#endif



#ifdef FS_STIK
    if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        is_stik = (nc && !(pf[0] & OUTLINE_CHAR));
    min = is_stik ? 2 : 3;
#else
    min = 3;
#endif

#ifdef FS_PSEUDO_BOLD
    is_pseudo = (STATE.bold_pct != 0) || (STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct != 0);
#endif

    if (np >= min)
    {
        /* count the number of types and points */
#ifdef FS_CFFR
        if (STATE.cur_lfnt->fnt_type == CFF_TYPE)
            count_points_cff(nc, ps, pe, pf, &n_types, &n_points);
        else
#endif
            count_points(_PS_ nc, ps, pe, pf, &n_types, &n_points);
        /* allocate an outline and set its pointers */
        outl = setup_outline(_PS_ nc, n_types, n_points);
        if (outl == 0)
            return NULL;

        /* fill it */
#ifdef FS_CFFR
        if (STATE.cur_lfnt->fnt_type == CFF_TYPE)
            fill_outline_cff(_PS_ nc, ps, pe, px, py, pf, outl);
        else
#endif
            fill_outline(_PS_ nc, ps, pe, px, py, pf, outl);

        /* set the bbox */
        set_bbox(n_points, outl);
    }
    else
    {
        /* it's a space character ... no points to count or set */
        FS_ULONG size = sizeof(FS_OUTLINE);
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_OUTLINE";
#endif
        outl = (FS_OUTLINE *) FSS_calloc(_PS_ size);
        if (outl == 0)
            return NULL;
        outl->size = size;
    }

    /* always use the (hinted) side bearing point from the TT data struct */
    if (STATE.flags & FLAGS_VERTICAL_ON)
    {
        FS_USHORT tsb, bsb;
        FS_LONG dx, dy;
        int i;

        tsb = np + TOPSIDEBEARING;
        bsb = np + BOTTOMSIDEBEARING;

        outl->dx = ( px[tsb] - px[bsb]) << 10;    /* not rounding */
        outl->dy = ( py[bsb] - py[tsb]) << 10;    /* not rounding */


        /* this moves the TOPSIDEBEARING to the origin  */
        /* the character will be centered on the origin */
        /* and the character will be BELOW the origin */
        dx = px[np + TOPSIDEBEARING] << 10;
        dy = py[np + TOPSIDEBEARING] << 10;

        /* but keep the adjustments integral */
        dx &= 0xFFFF0000;
        dy &= 0xFFFF0000;

        outl->lo_x -= dx;
        outl->hi_x -= dx;
        outl->lo_y -= dy;
        outl->hi_y -= dy;
        for (i = 0; i < n_points; i++)
        {
            outl->x[i] -= dx;
            outl->y[i] -= dy;
        }
    }
    else if (STATE.flags & FLAGS_VERTICAL_ROTATE_RIGHT_ON)
    {
        FS_USHORT lsb, rsb;
        FS_LONG temp, temp2 , dy;
        FS_FIXED A, D;
        int i;

        rsb = np + RIGHTSIDEBEARING;
        lsb = np + LEFTSIDEBEARING;

        /* right rotated advance based on horizontal advance */
        outl->dx = (py[rsb] - py[lsb]) << 10;    /* not rounding */
        outl->dy = (px[rsb] - px[lsb]) << 10;    /* not rounding */

        /* this centers the glyph based on scaled ascender, descender  */
        /* values, outline is shifted down by the amount |A|/2 - |D|/2 */
        /* Note that A is positive and D is negative in the hhea table */
        A = LongMulDiv(ttf->hhea->yAscender << 16,
                       STATE.lpm,
                       ttf->head->unitsPerEm);
        D = LongMulDiv(ttf->hhea->yDescender << 16,
                       STATE.lpm,
                       ttf->head->unitsPerEm);
        dy = (A >> 1) + (D >> 1);

        /* but keep the adjustments integral */
        dy &= 0xFFFF0000;

        outl->lo_y -= dy;
        outl->hi_y -= dy;

        /* rotate right by 90 degrees */
        temp  = outl->lo_x;
        temp2 = outl->hi_x;
        outl->lo_x = outl->lo_y;
        outl->lo_y = -temp2;

        outl->hi_x = outl->hi_y;
        outl->hi_y = -temp;

        for (i = 0; i < n_points; i++)
        {
            outl->y[i] -= dy;

            /* rotate right */
            temp = outl->x[i];
            outl->x[i] = outl->y[i];
            outl->y[i] =  -temp;
        }
    }
    else if (STATE.flags & FLAGS_VERTICAL_ROTATE_LEFT_ON)
    {
        FS_USHORT lsb, rsb;
        FS_LONG temp, temp2 , dy;
        FS_FIXED A, D;
        int i;

        rsb = np + RIGHTSIDEBEARING;
        lsb = np + LEFTSIDEBEARING;

        /* left rotated advance based on horizontal advance */
        outl->dx = ( py[lsb] - py[rsb]) << 10;    /* not rounding */
        outl->dy = ( px[lsb] - px[rsb]) << 10;    /* not rounding */

        /* this centers the glyph based on scaled ascender, descender  */
        /* values, outline is shifted down by the amount |A|/2 - |D|/2 */
        /* Note that A is positive and D is negative in the hhea table */
        A = LongMulDiv(ttf->hhea->yAscender << 16,
                       STATE.lpm,
                       ttf->head->unitsPerEm);
        D = LongMulDiv(ttf->hhea->yDescender << 16,
                       STATE.lpm,
                       ttf->head->unitsPerEm);
        dy = (A >> 1) + (D >> 1);

        /* but keep the adjustments integral */
        dy &= 0xFFFF0000;

        outl->lo_y -= dy;
        outl->hi_y -= dy;

        /* rotate left by 90 degrees */
        temp  = outl->lo_x;
        temp2 = outl->hi_x;
        outl->lo_x = -outl->hi_y;
        outl->hi_x = -outl->lo_y;
        outl->lo_y = temp;
        outl->hi_y = temp2;

        for (i = 0; i < n_points; i++)
        {
            outl->y[i] -= dy;

            /* rotate left */
            temp = outl->x[i];
            outl->x[i] = -outl->y[i];
            outl->y[i] =  temp;
        }
    }
    else /* horizontal writing */
    {
        FS_USHORT lsb, rsb;

        rsb = np + RIGHTSIDEBEARING;
        lsb = np + LEFTSIDEBEARING;

#ifdef FS_HINTS
        if (!(STATE.flags & FLAGS_HINTS_OFF))
        {
            if(STATE.cur_sfnt->senv->vanilla)
            {
                px[rsb] += 32;
                px[rsb] &= ~63;
                px[lsb] += 32;
                px[lsb] &= ~63;
            }
        }
#endif
        outl->dx = ( px[rsb] - px[lsb]) << 10;    /* not rounding */
        outl->dy = ( py[rsb] - py[lsb]) << 10;  /* not rounding */
    }


    /* set i_dx and i_dy if the side_bearing points are */
    /* integral after we apply the final transformation */
    /* Replace condition,  for based on dx / dy is zero */
    /* instead. jwd, 07/28/05.                          */

    if (outl->dx == 0 || outl->dy == 0)
    {
        /* rotation of N*90 degrees, integer escapements are useful */
        outl->i_dx = FS_ROUND(outl->dx); /* rounding here */
        outl->i_dy = FS_ROUND(outl->dy); /* rounding here */
    }
    else /* other rotations, integer escapments are not useful */
        outl->i_dx = outl->i_dy = 0;


#ifdef FS_PSEUDO_BOLD
    /* can't pseudo-bold stik fonts here because we don't */
    /* yet know the aspect ratio to use in <expand_stik> */
    /* which is required before we can call <embolden_outline> */
    if (is_pseudo && !is_stik)
    {
        if ( !(STATE.flags & FLAGS_ADD_WEIGHT))
        {
            if (np >= min)
            {
                FS_OUTLINE *tmp;
                tmp = embolden_outline(_PS_ outl, &n_types, &n_points);
                if (tmp != outl) /* passed back arg untouched - do NOT free it */
                    FSS_free_char(_PS_ outl);
                outl = tmp;
            }
            else
            {
                /* adjust the escapement only */
                int b;
                b = FS_ROUND((STATE.lpm * (STATE.bold_pct + STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)));
                if (b != 0)
                {
                    FIXED_VECTOR bold_vec;
                    bold_vec.x = STATE.cur_sfnt->user_scale[0];
                    bold_vec.y = STATE.cur_sfnt->user_scale[2];
                    fixed_norm(&bold_vec);
                    bold_vec.x *= b;
                    bold_vec.y *= b;
                    outl->dx = outl->dx + bold_vec.x;
                    outl->dy = outl->dy + bold_vec.y;
                    if (outl->i_dx)
                        outl->i_dx += FS_ROUND(bold_vec.x);
                    if (outl->i_dy)
                        outl->i_dy += FS_ROUND(bold_vec.y);
                }
            }
        }
    }
#endif /* FS_PSEUDO_BOLD */

    return outl;
}


/**
 * allocate an outline and set its pointers
 */
FS_OUTLINE *setup_outline(_DS_ FS_SHORT nc, FS_LONG n_types, FS_LONG n_points)
{
    FS_LONG size, type_off, x_off, y_off;
    FS_BYTE *p;
    FS_OUTLINE *outl;

    size = sizeof(FS_OUTLINE);
    FS_ALIGN(size);
    type_off = size;
    size += n_types;
    FS_ALIGN(size);
    x_off = size;
    size += (4 * n_points);
    FS_ALIGN(size);
    y_off = size;
    size += (4 * n_points);

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_OUTLINE";
#endif
    p = (FS_BYTE *)FSS_calloc(_PS_ size);
    if (p == 0)
        return 0;
    outl = (FS_OUTLINE *)p;
    outl->size = size;
    outl->nc = nc;
    outl->num = (FS_SHORT)n_types;
    outl->np = (FS_SHORT)n_points;
    outl->type = (FS_BYTE *)(p + type_off);
    outl->x = (FS_FIXED *)(p + x_off);
    outl->y = (FS_FIXED *)(p + y_off);
    outl->polarity = 1;
    return outl;
}

static FS_LONG
count_outline_points(FS_OUTLINE *outl, FS_BOOLEAN user_mem)
{
    FS_LONG n_points;
    FS_BYTE *t_src;
    int i;

    if (!outl) return 0;

    n_points = 0;

    if (user_mem)
    {
        t_src = outl->type;
    }
    else
    {
        t_src = (FS_BYTE *)(outl->type);
    }
    for (i = 0; i < outl->num; i++)
    {
        switch (*t_src++ & 0x7f)
        {
        case FS_MOVETO:
        case FS_LINETO:
            n_points += 1;
            break;
        case FS_QUADTO:
            n_points += 2;
            break;
        case FS_CUBETO:
            n_points += 3;
            break;
        default:
            break;
        }
    }
    return n_points;
}

FS_OUTLINE *
copy_outline(_DS_ FS_OUTLINE *outl,
             FS_BOOLEAN user_mem /* source is in user mem vs FS_SERVER mem */)
{
    FS_LONG n_points;
    FS_OUTLINE *nol;

    if (!outl)
        return NULL;

    n_points = count_outline_points(outl, user_mem);

    nol = setup_outline(_PS_ outl->nc, outl->num, n_points);
    if (!nol)
    {
        return NULL;
    }
    /* "setup_outline()" sets:
        size
        nc
        num
        type pointer (not elements)
        x pointer (not elements)
        y pointer (not elements)
        polarity = 1
    */
    /* copy other data */

    nol->cache_ptr = NULL;
    nol->lo_x = outl->lo_x;
    nol->hi_x = outl->hi_x;
    nol->lo_y = outl->lo_y;
    nol->hi_y = outl->hi_y;
    nol->i_dx = outl->i_dx;
    nol->i_dy = outl->i_dy;
    nol->dx = outl->dx;
    nol->dy = outl->dy;
    nol->polarity = outl->polarity;

    if (user_mem)
    {
        /* copy the curve elements */
        {
            FS_BYTE *b;
            b = (FS_BYTE *)(nol->type);
            if (b) /* these shenanigans remove Linux "possible NULL" warning */
            {
                SYS_MEMCPY(b, outl->type, outl->num * sizeof(FS_BYTE));
            }
        }

        /* copy the points */
        {
            FS_FIXED *f;
            f = (FS_FIXED *)(nol->x);
            if (f) /* these shenanigans remove Linux "possible NULL" warning */
            {
                SYS_MEMCPY(f, outl->x, n_points * sizeof(FS_FIXED));
            }
            f = (FS_FIXED *)(nol->y);
            if (f) /* these shenanigans remove Linux "possible NULL" warning */
            {
                SYS_MEMCPY(f, outl->y, n_points * sizeof(FS_FIXED));
            }
        }
    }
    else /* shared mem */
    {
        /* copy the curve elements */
        {
            FS_BYTE *d, *s;
            d = (FS_BYTE *)(nol->type);
            s = (FS_BYTE *)(outl->type);
            if (d && s)        /* this removes Linux "possible NULL" warning */
            {
                SYS_MEMCPY(d, s, outl->num * sizeof(FS_BYTE));
            }
        }

        /* copy the points */
        {
            FS_FIXED *d, *s;
            d = (FS_FIXED *)(nol->x);
            s = (FS_FIXED *)(outl->x);
            if (d && s)        /* this removes Linux "possible NULL" warning */
            {
                SYS_MEMCPY(d, s, n_points * sizeof(FS_FIXED));
            }
            d = (FS_FIXED *)(nol->y);
            s = (FS_FIXED *)(outl->y);
            if (d && s)        /* this removes Linux "possible NULL" warning */
            {
                SYS_MEMCPY(d, s, n_points * sizeof(FS_FIXED));
            }
        }
    }
    return nol;
}

FS_OUTLINE *find_or_make_outline(_DS_ LFNT *lfnt, SFNT *sfnt,
                                 FS_ULONG id, FS_ULONG index)
{
    FS_OUTLINE *outl;

#ifdef FS_CACHE_OUTLINES
    if (!(STATE.flags & FLAGS_RTGAH_REF) )
        outl = find_outline_in_cache(_PS_ index);
    else
        outl = 0;

    if (!outl)
#endif
    {
#ifdef FS_STIK
#if !defined(FS_USE_REFLINES)
        if ( (lfnt->fontflags & FONTFLAG_STIK) &&
                !(lfnt->fontflags & FONTFLAG_DIRECT_DRAW_ON))
        {
            FS_ULONG uid = id;
            if (STATE.flags & FLAGS_CMAP_OFF)
            {
                FS_USHORT platform = STATE.platform;
                FS_USHORT encoding = STATE.encoding;
                FSS_set_cmap(_PS_ 3, 1);
                uid = FSS_inverse_map_char(_PS_ index);
                FSS_set_cmap(_PS_ platform, encoding);
            }
            if ((uid >= 0x1100 && uid <= 0x11ff) || (uid >= 0x2e80 && uid <= 0x2fdf)  ||
                    (uid >= 0x3040 && uid <= 0x31bf) || (uid >= 0x3200 && uid <= 0x4dbf)  ||
                    (uid >= 0x4e00 && uid <= 0x9faf) || (uid >= 0xa000 && uid <= 0xa4cf)  ||
                    (uid >= 0xac00 && uid <= 0xd7af) || (uid >= 0xf900 && uid <= 0xfaff)  ||
                    (uid == 0xffff) )
                STATE.use_reflines = false;
            else
                STATE.use_reflines = true;
        }
        else
            STATE.use_reflines = false;
#else
        STATE.use_reflines = true;
#endif /* FS_USE_REFLINES */
#else
        id = id;      /* avoid compiler warnings when FS_STIK is not defined */
        lfnt = lfnt;
#endif /* FS_STIK */

        outl = make_outline(_PS_ sfnt, index);
        if (STATE.error || !outl)
            return 0;

        outl->outl_flag = 0;
        outl->outl_flag |= STATE.any_hints;
        if (STATE.flags & FLAGS_FIX_DROPOUTS)
            outl->outl_flag |= OUTL_FLAGS_SCANCTRL;

#ifdef FS_CACHE_OUTLINES
        if ( !(STATE.flags & FLAGS_RTGAH_REF) )
            save_outline_to_cache(_PS_ index, outl);
#endif
    }
#ifdef FS_CACHE_OUTLINES
    else
    {
        STATE.any_hints = (outl->outl_flag & OUTL_FLAGS_ANYHINTS) |
                          (outl->outl_flag & OUTL_FLAGS_RTGAH);
        if (outl->outl_flag & OUTL_FLAGS_SCANCTRL)
            STATE.flags |= FLAGS_FIX_DROPOUTS;
        else
            STATE.flags &= ~FLAGS_FIX_DROPOUTS;

#ifdef FS_EDGE_RENDER

        STATE.any_hints |= (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_BAZ) |
                           (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_MAZ);

#endif
#ifdef FS_STIK
#if !defined(FS_USE_REFLINES)
        if ( (lfnt->fontflags & FONTFLAG_STIK) &&
                !(lfnt->fontflags & FONTFLAG_DIRECT_DRAW_ON))
        {
            FS_ULONG uid = id;
            if (STATE.flags & FLAGS_CMAP_OFF)
            {
                FS_USHORT platform = STATE.platform;
                FS_USHORT encoding = STATE.encoding;
                FSS_set_cmap(_PS_ 3, 1);
                uid = FSS_inverse_map_char(_PS_ index);
                FSS_set_cmap(_PS_ platform, encoding);
            }
            if ((uid >= 0x1100 && uid <= 0x11ff) || (uid >= 0x2e80 && uid <= 0x2fdf)  ||
                    (uid >= 0x3040 && uid <= 0x31bf) || (uid >= 0x3200 && uid <= 0x4dbf)  ||
                    (uid >= 0x4e00 && uid <= 0x9faf) || (uid >= 0xa000 && uid <= 0xa4cf)  ||
                    (uid >= 0xac00 && uid <= 0xd7af) || (uid >= 0xf900 && uid <= 0xfaff)  ||
                    (uid == 0xffff) )
                STATE.use_reflines = false;
            else
                STATE.use_reflines = true;
        }
        else
            STATE.use_reflines = false;
#else
        STATE.use_reflines = true;
#endif /* FS_USE_REFLINES */
#endif /* FS_STIK */
    }
#endif

#ifdef FS_EDGE_RENDER
    STATE.adfGridFitType = ADF_GRID_FIT_NONE;
    if ((STATE.flags & FLAGS_MAZ_ON)) /* user override */
        STATE.adfGridFitType = ADF_GRID_FIT_MAZ_PIXEL;
    else
    {
        if (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_MAZ)
            STATE.adfGridFitType = ADF_GRID_FIT_MAZ_PIXEL;
        else if (outl->outl_flag & OUTL_FLAGS_GRIDFITTYPE_BAZ)
            STATE.adfGridFitType = ADF_GRID_FIT_BAZ_PIXEL;
    }
#endif

    return outl;
}

/****************************************************************/
FS_OUTLINE *internal_get_outline(_DS_ FS_ULONG id)
{
    SFNT *sfnt;
    LFNT *lfnt;
    FS_ULONG index;
    FS_OUTLINE *outl;

    /* character id to index */
    index = map_char(_PS_ id, 0);
    if (STATE.error)
        return 0;

    if (check_sfnt(_PS0_))
        return 0;

    lfnt = STATE.cur_lfnt;
    sfnt = STATE.cur_sfnt;
#ifdef FS_HINTS
    if ((lfnt->fontflags & FONTFLAG_STIK) != FONTFLAG_STIK &&
            (lfnt->fnt_type != PFR_TYPE) &&
            (sfnt->ref_lines[RTGA_BEEN_THERE] == 0))
        get_ref_lines(_PS0_); /* get RTGAH ref lines since not already done */
#endif
    outl = find_or_make_outline(_PS_ lfnt, sfnt, id, index);

    return outl;
}
#ifdef FS_FOR_FUTURE_USE
/*****************************************************************/
/* rotate outline according to sine and cosine of rotation angle */
FS_VOID rotate_outline(_DS_ FS_OUTLINE *outl, FS_FIXED sine, FS_FIXED cosine)
{
    FS_LONG i, n_points;
    
    /* fast out if no rotation */
    if( cosine == 65536 ) 
        return;


    n_points = count_outline_points(_PS_ outl, 0);

    /* apply rotation transform to all points */
    for(i=0; i<n_points ; i++)
    {
        FS_FIXED x = outl->x[i];
        FS_FIXED y = outl->y[i];
        outl->x[i] = FixMul(x,cosine) + FixMul(y,sine);
        outl->y[i] = FixMul(y,cosine) - FixMul(x,sine);
    }

    return;
}
#endif

/****************************************************************/
/* shift outline by a quarter pixel amounts in x and y          */
FS_VOID shift_outline(FS_OUTLINE *outl, FS_SHORT cspx, FS_SHORT cspy)
{
    FS_FIXED shift;
    FS_LONG i, n_points;
    
    if (cspx == 0 && cspy == 0) 
        return;
            
    n_points = count_outline_points(outl, 0);

    if (cspx != 0)
    {
        /* quantize x shift */
        shift = cspx * 16384;

        /* add shift amount to all x values */
        for(i=0; i<n_points ; i++)
        {
            outl->x[i] += shift;
        }
    }

    if (cspy != 0) 
    {
        /* quantize y shift */
        shift = cspy * 16384;

        /* add shift amount to all y values */
        for(i=0; i<n_points ; i++)
        {
            outl->y[i] -= shift;
        }
    }
    return;
}

/****************************************************************/
/* to support GPOS table, need the coords of original points */
FS_LONG FSS_get_gpos_pts(_DS_ FS_USHORT id, FS_USHORT num, FS_USHORT *pts, FS_LONG *x, FS_LONG *y)
{
    SFNT *sfnt;
    FS_USHORT index;
    FS_ULONG flags = STATE.flags;
    FS_BOOLEAN useHints;
    fsg_SplineKey *key;

    STATE.flags |= FLAGS_CMAP_OFF;
    index = map_char(_PS_ id, 0);
    STATE.flags = flags;

    if (STATE.error)
    {
        return STATE.error;
    }
    if (check_sfnt(_PS0_))
    {
        return STATE.error;
    }
    sfnt = STATE.cur_sfnt;

    key = (FS_VOID *)(sfnt->senv->ttkey);

    /* is maxp being nullified by unload_ttf(), if so release
     * the key and senv buffers and re-scale the sfnt
     */
    if (!key->maxp)
    {
        FS_FIXED *s;
        FS_FIXED xppm, yppm, skew, s10;

        s = sfnt->user_scale;

        xppm = s[0];
        skew = s[1];
        s10  = s[2];
        yppm = s[3];

        delete_key(_PS_ key);

        FSS_free(_PS_ sfnt->senv);
        sfnt->senv = 0;

        scale_font_ttf(_PS_ sfnt, xppm, skew, s10, yppm);
        if (STATE.error)
        {
            return STATE.error;
        }
        key = (FS_VOID *)(sfnt->senv->ttkey);
    }
    key->glyphIndex = index;

#ifdef FS_HINTS
    useHints = !(STATE.flags & FLAGS_HINTS_OFF);
#else
    useHints = 0;
#endif

    if (STATE.cur_lfnt->fnt_type != CFF_TYPE)
        fsg_GridFit(_PS_ key, useHints);
#ifdef FS_CFFR
    else
        cff_GridFit(_PS_ key, useHints);
#endif

    if (STATE.error)
    {
        return STATE.error;
    }

    /* copy the requested data */
    {
        fnt_ElementType *elementPtr;
        FS_BYTE *workSpacePtr;
        fsg_OffsetInfo *offsetPtr;
        FS_LONG *px, *py;
        FS_LONG dx = 0, dy = 0;
        FS_SHORT *pe;
        FS_SHORT nc, np;
        FS_SHORT i, n;

        /* find the outline */
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
        workSpacePtr = (FS_BYTE *)(STATE.server->workspace);
        offsetPtr = &(key->elementInfoRec.offsets[GLYPHELEMENT]);

        /* x coords, y coords, contour endpoints, number of contours and number of points*/
        px = (FS_LONG *)(workSpacePtr + offsetPtr->newXOffset);
        py = (FS_LONG *)(workSpacePtr + offsetPtr->newYOffset);
        pe = (FS_SHORT *)(workSpacePtr + offsetPtr->endPointOffset);
        nc = elementPtr->nc;
        np = nc ? 1 + pe[nc - 1] : 0;

        if (STATE.flags & FLAGS_VERTICAL_ON)
        {
            /* this moves the TOPSIDEBEARING to the origin  */
            /* the character will be centered on the origin */
            /* and the character will be BELOW the origin */
            dx = px[np + TOPSIDEBEARING] << 10;
            dy = py[np + TOPSIDEBEARING] << 10;

            /* but keep the adjustments integral */
            dx &= 0xFFFF0000;
            dy &= 0xFFFF0000;
        }

        if (STATE.flags & FLAGS_VERTICAL_ROTATE_RIGHT_ON)
        {
            for (i = 0; i < num; i++)
            {
                n = pts[i];
                if (n >= np)
                {
                    return STATE.error = POINTS_DATA_ERR;
                }
                /* rotate right */
                x[i] =  py[n];
                y[i] = -px[n];
            }
        }
        else if (STATE.flags & FLAGS_VERTICAL_ROTATE_LEFT_ON)
        {
            for (i = 0; i < num; i++)
            {
                n = pts[i];
                if (n >= np)
                {
                    return STATE.error = POINTS_DATA_ERR;
                }
                /* rotate left */
                x[i] = -py[n];
                y[i] =  px[n];
            }
        }
        else /* horizontal or regular vertical writing */
        {
            for (i = 0; i < num; i++)
            {
                n = pts[i];
                if (n >= np)
                {
                    return STATE.error = POINTS_DATA_ERR;
                }
                x[i] = px[n] - dx;
                y[i] = py[n] - dy;
            }
        }
    }

    return SUCCESS;
}

/****************************************************************/

/* make the scaled outline for gIndex */
FS_OUTLINE *make_outline_ttf(_DS_ SFNT *sfnt, FS_ULONG gIndex)
{
    FS_OUTLINE *outl;
    FS_BOOLEAN useHints;
    fsg_SplineKey *key;

#if defined(FS_EDGE_HINTS) || defined(FS_EDGE_RENDER)
    FS_USHORT EdgeHintType = 0;
#endif

    key = (FS_VOID *)(sfnt->senv->ttkey);

    /* is maxp being nullified by unload_ttf(), if so release
     * the key and senv buffers and re-scale the sfnt
     */
    if (!key->maxp)
    {
        FS_FIXED *s;
        FS_FIXED xppm, yppm, skew, s10;

        s = sfnt->user_scale;

        xppm = s[0];
        skew = s[1];
        s10  = s[2];
        yppm = s[3];

        delete_key(_PS_ key);

        FSS_free(_PS_ sfnt->senv);
        sfnt->senv = 0;

        scale_font_ttf(_PS_ sfnt, xppm, skew, s10, yppm);
        if (STATE.error)
        {
            return NULL;
        }
        key = (FS_VOID *)(sfnt->senv->ttkey);
    }

#if defined(FS_EDGE_HINTS) || defined(FS_EDGE_RENDER)
    {
        TTF *ttf = (TTF *)(sfnt->lfnt->fnt);
        if (ttf->adfh) /* is the ADF hint table there? */
        {
            FS_SHORT noCenter, cvtstart, numylines, isrighttoleft;

            /* get the ADF hints from the adfh table */
            adfGetEdgeHintData(_PS_ ttf, gIndex, &EdgeHintType,
                               &noCenter, &cvtstart, &numylines, &isrighttoleft);
#ifdef FS_EDGE_HINTS
            key->globalGS.maz_data.gridFitType = EdgeHintType;
            if ((STATE.flags & FLAGS_MAZ_ON)) /* user override */
                key->globalGS.maz_data.gridFitType = ADF_GRID_FIT_MAZ_PIXEL;
            key->globalGS.maz_data.noCenter = noCenter;
            key->globalGS.maz_data.cvtstart = cvtstart;
            key->globalGS.maz_data.numylines = numylines;
            key->globalGS.maz_data.isrighttoleft = isrighttoleft;
#endif
#ifdef FS_EDGE_RENDER
            STATE.adfGridFitType = EdgeHintType;
            if ((STATE.flags & FLAGS_MAZ_ON)) /* user override */
                STATE.adfGridFitType = ADF_GRID_FIT_MAZ_PIXEL;
#endif
        }
        else
        {
#ifdef FS_EDGE_HINTS
            key->globalGS.maz_data.gridFitType = ADF_GRID_FIT_NONE;
#endif
#ifdef FS_EDGE_RENDER
            STATE.adfGridFitType = ADF_GRID_FIT_NONE;
#endif
        }
    }
#endif /* FS_EDGE_HINTS or FS_EDGE_RENDER */

    if (gIndex >= key->maxp->numGlyphs)
        gIndex = 0; /* substitute missing char index */

    key->glyphIndex = (FS_SHORT)gIndex;    /* at least a FS_USHORT ? */

#ifdef FS_HINTS
    useHints = !(STATE.flags & FLAGS_HINTS_OFF);
#else
    useHints = 0;
#endif

#ifdef FS_TRACE
    FS_PRINTF(("****************** making char=%d **********************\n", gIndex));
    in_char = 1;
    indent_is = 0;

#endif

    if (STATE.cur_lfnt->fnt_type != CFF_TYPE)
        fsg_GridFit(_PS_ key, useHints);
#ifdef FS_CFFR
    else
        cff_GridFit(_PS_ key, useHints);
#endif


#ifdef FS_TRACE
    in_char = 0;
#endif
    if (STATE.error)
        return 0;

    outl = extract_outline(_PS_ key);

#ifdef FS_STATS
    if (outl) made_outl++;
#endif

    /* if the user has set FLAG_DROPOUTS_ON ... we WILL fix them */
    if (STATE.flags & FLAGS_DROPOUTS_ON)
        STATE.flags |= FLAGS_FIX_DROPOUTS;
    else
    {
#ifdef FS_HINTS
        /* otherwise we do what the font wants when FS_HINTS is defined */
        if (get_dropout_control(key) || STATE.cur_lfnt->fnt_type == CFF_TYPE)
            STATE.flags |= FLAGS_FIX_DROPOUTS;
        else
#endif
            STATE.flags &= ~FLAGS_FIX_DROPOUTS;
    }
    return outl;
}

/****************************************************************/
#ifdef FS_DUMP
FS_VOID dump_outline(FS_OUTLINE *outl)
{
    int i;
    FS_FIXED *x, *y;
    FILECHAR proc_str[21];

    if (!outl)
        return;

    proc_str[0] = '\0';

    /* header */
    FS_PRINTF(("%sxmin=%12.5f xmax=%12.5f ymin=%12.5f ymax=%12.5f\n",
               proc_str,
               outl->lo_x / 65536.0, outl->hi_x / 65536.0,
               outl->lo_y / 65536.0, outl->hi_y / 65536.0));

    FS_PRINTF(("%si_dx=%d i_dy=%d dx=%12.5f dy=%12.5f nc=%d num=%d np=%d\n",
               proc_str, outl->i_dx, outl->i_dy, outl->dx / 65536.0, outl->dy / 65536.0,
               outl->nc, outl->num, outl->np));

    /* now the curve data */
    x = outl->x;
    y = outl->y;
    for (i = 0; i < outl->num; i++)
    {
        switch (outl->type[i] & 0x7f)
        {
        case FS_MOVETO:
            FS_PRINTF(("%s%12.5f %12.5f moveto\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            break;
        case FS_LINETO:
            FS_PRINTF(("%s%12.5f %12.5f lineto\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            break;
        case FS_QUADTO:
            FS_PRINTF(("%s%12.5f %12.5f\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            FS_PRINTF(("%s%12.5f %12.5f quadto\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            break;
        case FS_CUBETO:
            FS_PRINTF(("%s%12.5f %12.5f\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            FS_PRINTF(("%s%12.5f %12.5f\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            FS_PRINTF(("%s%12.5f %12.5f cubeto\n", proc_str, *x++ / 65536.0, *y++ / 65536.0));
            break;
        default:
            break;
        }
    }
    FS_PRINTF(("\n"));
}

#endif /* FS_DUMP */

/****************************************************************/
/* MACRO's make the following function a little more readable */
/* append points and types ... converting from F26DOT6 to FS_FIXED */
#define _MOVETO(x0,y0) \
    *type++ = FS_MOVETO; *x++ = (x0)<<10; *y++ = (y0)<<10

#define _LINETO(x0,y0) \
    *type++ = FS_LINETO; *x++ = (x0)<<10; *y++ = (y0)<<10

#define _QUADTO(x0,y0,x1,y1)\
    *type++ = FS_QUADTO; *x++ = (x0)<<10; *y++ = (y0)<<10;\
    *x++ = (x1)<<10; *y++ = (y1)<<10
#define _CUBETO(x0,y0,x1,y1,x2,y2)\
    *type++ = FS_CUBETO; *x++ = (x0)<<10; *y++ = (y0)<<10;\
    *x++ = (x1)<<10; *y++ = (y1)<<10; *x++ = (x2)<<10; *y++ = (y2)<<10

#define _SPLINETO(x0,y0,x1,y1)\
    *type++ = FS_QUADTO; *x++ = (x0)<<10; *y++ = (y0)<<10;\
    *x++ = ((x0)+(x1))<<9; *y++ = ((y0)+(y1))<<9

/****************************************************************/

#ifdef FS_CFFR
/*lint -e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */
FS_VOID fill_outline_cff(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_LONG *px, FS_LONG *py, FS_BYTE *pf, FS_OUTLINE *outl)
{
    FS_SHORT ctr, thisPoint, startPoint, endPoint;
    FS_BYTE *onC, thisOnC;
    FS_LONG *xx, *yy, tx, ty, prX = 0, prY = 0;
    FS_LONG pprX = 0, pprY = 0;
    FS_BYTE *type = (FS_BYTE *)outl->type;
    FS_LONG *x = (FS_FIXED *)outl->x;
    FS_LONG *y = (FS_FIXED *)outl->y;
    int is_outline = 1;

    if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        is_outline = (nc && (pf[0] & OUTLINE_CHAR));

    outl->nc = 0;
    for (ctr = 0; ctr < nc; ctr++)
    {
        endPoint = pe[ctr];
        startPoint = ps[ctr];
        xx = &(px[startPoint]);
        yy = &(py[startPoint]);
        onC = &(pf[startPoint]);
        if (startPoint == endPoint)
            continue;
        outl->nc++;

        tx = *xx++;
        ty = *yy++;
        thisOnC = (*onC++);


        /* the initial MOVETO */
        _MOVETO(tx, ty);
        thisPoint = startPoint + 1;

        /* remaining lines and quads */
        for ( ; thisPoint <= endPoint; thisPoint++)
        {
            tx = *xx++;
            ty = *yy++;
            thisOnC = (*onC++);

            if (thisOnC)
            {
                _LINETO(tx, ty);
            }
            else
            {
                pprX = tx;
                pprY = ty;
                tx = *xx++;
                ty = *yy++;
                thisPoint++;
                thisOnC = (*onC++);
                if (thisOnC)
                {
                    /* shouldn't happen */
                    _QUADTO(pprX, pprY, tx, ty);
                }
                else
                {
                    prX = tx;
                    prY = ty;
                    tx = *xx++;
                    ty = *yy++;
                    thisPoint++;
                    thisOnC = (*onC++);
                    if (thisOnC)
                    {
                        _CUBETO(pprX, pprY, prX, prY, tx, ty);
                    }
                    else
                    {
                        /* shouldn't happen */
                    }
                }

            }
        } /* for each remaining point */
    } /* for each contour */

    /* make sure we set the "OUTLINE_CHAR" bit as needed */
    if (is_outline && outl->num)
        outl->type[0] |= OUTLINE_CHAR;
}
/*lint -e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */
#endif /* FS_CFFR */

/****************************************************************/
FS_VOID fill_outline(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_LONG *px, FS_LONG *py, FS_BYTE *pf, FS_OUTLINE *outl)
{
    FS_SHORT ctr, thisPoint, startPoint, endPoint;
    FS_BYTE *onC, thisOnC, firstOnC, secondOnC = 0, prevOnC;
    FS_LONG *xx, *yy, tx, ty, firstX, firstY, secondX = 0, secondY = 0, prX = 0, prY = 0;
    FS_BYTE *type = (FS_BYTE *)(outl->type);
    FS_LONG *x = (FS_FIXED *)(outl->x);
    FS_LONG *y = (FS_FIXED *)(outl->y);
    int is_outline = 1;
    int connect = 0;
    FS_LONG x0, y0;   /* the first ONCURVE point in a contour */

    if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        is_outline = (nc && (pf[0] & OUTLINE_CHAR));

    outl->nc = 0;
    for (ctr = 0; ctr < nc; ctr++)
    {
        endPoint = pe[ctr];
        startPoint = ps[ctr];
        xx = &(px[startPoint]);
        yy = &(py[startPoint]);
        onC = &(pf[startPoint]);
        if (startPoint == endPoint)
            continue;
        outl->nc++;

        /* in a STIK char, if the last point in the
        * contour is OFF_CURVE, we need to connect
        * back to the contour's start point
        */
        if (!is_outline)
            connect = 0 == (pf[pe[ctr]] & 1);

        /* the initial MOVETO */
        firstOnC = (*onC++) & 1;
        firstX = *xx++;
        firstY = *yy++;
        if (firstOnC)
        {
            _MOVETO(firstX, firstY);
            thisPoint = startPoint + 1;
            prevOnC = firstOnC;
        }
        else
        {
            secondOnC = (*onC++);
            secondX = *xx++;
            secondY = *yy++;
            prevOnC = secondOnC;
            thisPoint = startPoint + 2;
            if (secondOnC)
            {
                _MOVETO(secondX, secondY);
            }
            else
            {
                _MOVETO((firstX + secondX) >> 1, (firstY + secondY) >> 1);
                prX = secondX;
                prY = secondY;
            }
        }
        /* capture initial ONCURVE point */
        x0 = x[-1];
        y0 = y[-1];

        /* remaining lines and quads */
        for ( ; thisPoint <= endPoint; thisPoint++)
        {
            tx = *xx++;
            ty = *yy++;
            thisOnC = (*onC++);
            if (!prevOnC)
            {
                if (thisOnC)
                {
                    _QUADTO(prX, prY, tx, ty);
                }
                else
                {
                    _SPLINETO(prX, prY, tx, ty);
                }
            }
            else
            {
                if (thisOnC)
                {
                    _LINETO(tx, ty);
                }
            }
            prevOnC = thisOnC;
            prX = tx;
            prY = ty;
        }

        if (is_outline || connect)
        {
            /* make sure we connect to <firstX,firstY> */
            if (firstOnC)
            {
                if (prevOnC)
                {
                    _LINETO(firstX, firstY);
                }
                else
                {
                    _QUADTO(prX, prY, firstX, firstY);
                }
            }
            else
            {
                if (!prevOnC)
                {
                    _SPLINETO(prX, prY, firstX, firstY);
                }
                if (secondOnC)
                {
                    _QUADTO(firstX, firstY, secondX, secondY);
                }
                else
                {
                    _SPLINETO(firstX, firstY, secondX, secondY);
                }
            }
            /* make sure LAST==FIRST, rounding errors sometimes happen */
            x[-1] = x0;
            y[-1] = y0;
        }
    }

    /* make sure we set the "OUTLINE_CHAR" bit as needed */
    if (is_outline && outl->num)
        outl->type[0] |= OUTLINE_CHAR;
}

#undef _MOVETO
#undef _LINETO
#undef _QUADTO
#undef _CUBETO
#undef _SPLINETO

/****************************************************************/
/* count the segment types and points -- same structure as fill_outline() */
FS_VOID count_points(_DS_ FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_BYTE *pf, FS_LONG *_types, FS_LONG *_points)
{
    FS_SHORT ctr, thisPoint, startPoint, endPoint;
    FS_BYTE *onC, thisOnC, firstOnC, secondOnC = 0, prevOnC;
    FS_LONG types = 0, points = 0;
    int is_outline = 1;
    int connect = 0;

    /* FS_LONG firstX,firstY,secondX,secondY,prX,prY; */

    /* FS_BYTE *type = outl->type; */
    /* FS_LONG *x = outl->x; */
    /* FS_LONG *y = outl->y; */

    if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        is_outline = (nc && (pf[0] & OUTLINE_CHAR));

    for (ctr = 0; ctr < nc; ctr++)
    {
        endPoint = pe[ctr];
        startPoint = ps[ctr];
        /* xx = &(px[startPoint]); */
        /* yy = &(py[startPoint]); */
        onC = &(pf[startPoint]);
        if (startPoint == endPoint)
            continue;
        /* in a STIK char, if the last point in the
        * contour is OFF_CURVE, we need to connect
        * back to the contour's start point
        */
        if (!is_outline)
            connect = 0 == (pf[pe[ctr]] & 1);

        /* the initial moveto */
        firstOnC = (*onC++) & 1;
        /* firstX = *xx++ + xoffset; */
        /* firstY = *yy++ + yoffset; */
        if (firstOnC)
        {
            /* _MOVETO(firstX, firstY); */
            types++;
            points++;
            thisPoint = startPoint + 1;
            prevOnC = firstOnC;
        }
        else
        {
            secondOnC = (*onC++) & 1;
            /* secondX = *xx++ + xoffset; */
            /* secondY = *yy++ + yoffset; */
            prevOnC = secondOnC;
            thisPoint = startPoint + 2;
            if (secondOnC)
            {
                /* _MOVETO(secondX, secondY); */
                types++;
                points++;
            }
            else
            {
                /* _MOVETO(firstX + secondX) >> 1, (firstY + secondY) >> 1); */
                types++;
                points++;
                /* prX = secondX; */
                /* prY = secondY; */
            }
        }

        /* remaining lines and quads */
        for ( ; thisPoint <= endPoint; thisPoint++)
        {
            /* tx = *xx++; */
            /* ty = *yy++; */
            thisOnC = (*onC++) & 1;
            if (!prevOnC)
            {
                if (thisOnC)
                {
                    /* _QUADTO(prX, prY, tx, ty); */
                    types++;
                    points += 2;
                }
                else
                {
                    /* _SPLINETO(prX, prY, tx, ty); */
                    types++;
                    points += 2;
                }
            }
            else
            {
                if (thisOnC)
                {
                    /* _LINETO(tx, ty); */
                    types++;
                    points++;
                }
            }
            prevOnC = thisOnC;
            /* prX = tx; */
            /* prY = ty; */
        }

        if (is_outline || connect)
        {
            /* make sure we connect to <firstX,firstY> */
            if (firstOnC)
            {
                if (prevOnC)
                {
                    /* _LINETO(firstX, firstY); */
                    types++;
                    points++;
                }
                else
                {
                    /* _QUADTO(prX, prY, firstX, firstY); */
                    types++;
                    points += 2;
                }
            }
            else
            {
                if (!prevOnC)
                {
                    /* _SPLINETO(prX, prY, firstX, firstY); */
                    types++;
                    points += 2;
                }
                if (secondOnC)
                {
                    /* _QUADTO(firstX, firstY, secondX, secondY); */
                    types++;
                    points += 2;
                }
                else
                {
                    /* _SPLINETO(firstX, firstY, secondX, secondY); */
                    types++;
                    points += 2;
                }
            }
        }
    }
    *_types = types;
    *_points = points;
}

/****************************************************************/
#ifdef FS_CFFR
/*lint -e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */
/* count the segment types and points -- same structure as fill_outline() */
static FS_VOID count_points_cff(FS_SHORT nc, FS_SHORT *ps, FS_SHORT *pe, FS_BYTE *pf, FS_LONG *_types, FS_LONG *_points)
{
    FS_SHORT ctr, thisPoint, startPoint, endPoint;
    FS_BYTE *onC, thisOnC;
    FS_LONG types = 0, points = 0;

    for (ctr = 0; ctr < nc; ctr++)
    {
        endPoint = pe[ctr];
        startPoint = ps[ctr];
        onC = &(pf[startPoint]);
        if (startPoint == endPoint)
            continue;

        /* remaining lines and cubics */
        for (thisPoint = startPoint; thisPoint <= endPoint; thisPoint++)
        {
            /* tx = *xx++; */
            /* ty = *yy++; */
            thisOnC = (*onC++) & 1;
            if (thisOnC)
            {
                /* _LINETO(prX, prY, tx, ty); */
                types++;
                points += 1;
            }
            else
            {
                /* _CUBETO(prX, prY, tx, ty); */
                types++;
                points += 3;
                thisPoint += 2;
                onC += 2;
            }
        }
    }
    *_types = types;
    *_points = points;
}
/*lint +e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */
#endif

#ifdef FS_CONTOUR_WINDING_DETECTION
/****************************************************************/
/* The checking code builds an array indexed by contour number and
 * having values 0 if ink is on the right and 1 if ink is on the left.
 */
FS_LONG check_contour_winding(_DS_ FS_OUTLINE *outl)
{
    FS_ULONG saved_flags;
    FS_FIXED saved_bold_pct;
    FS_USHORT isWrong;
    FS_USHORT wrongCt, rightCt;

    FS_SHORT nc;  /* number of contours in the outline */
    FS_SHORT currentLoop;
    FS_LONG i;
    TNODE *nodes;

    /* Disable dropout control, hint processing and pseudo bold for
    * contour winding detection. Restore when we are done */
    saved_flags    = STATE.flags;
    saved_bold_pct = STATE.bold_pct;
    STATE.bold_pct = 0;
    STATE.flags   |= FLAGS_HINTS_OFF;
    STATE.flags   &= FLAGS_DROPOUTS_OFF;

    nc = outl->nc;

    make_bitmap(_PS_ outl);   /* generate the transition list, */
    /* but don't really make the bitmap */
    if (STATE.error)
    {
        FSS_free_char(_PS_ outl);
        return 0;
    }

    /* Analyze the tlist */
    {
        TLIST *list;
        FS_LONG index, winding;
        FS_FIXED t;
        int which[2] = { -1, 1};
        TNODE *p, *p0;

        STATE.error = SUCCESS;

        list = (TLIST *)(STATE.server->tlist);
        nodes = (TNODE *)(list->nodes);

        for (i = 0; i < list->num; i++)
        {
            index = list->indices[i];

            /* process a row of intercepts */
            while (index >= 0)
            {
                /* get next transition */
                p0 = p = nodes + index;
                index = p->next;
                if (index < 0)
                {
                    STATE.error = ERR_NO_MATCHING;
                    break;    /* no next element -- can't get a matching pair */
                }
                t = p->value;
                winding = which[t & 1];
                isWrong = (winding == -1);  /* we are starting with an */
                /* off transition, rather than an on */

                /* get the matching transition -- most probably 1 iteration */
                do
                {
                    p = nodes + index;
                    index = p->next;
                    t = p->value;
                    p->value = 0;   /* mark trans that don't change beam as OK */
                    winding += which[t & 1];
                }
                while (winding != 0 && index >= 0);

                if (winding != 0)
                {
                    STATE.error = ERR_NO_MATCHING;
                    break;    /* didn't get a matching pair */
                }

                /* mark the on and off transitions */
                p0->value =    p->value = isWrong;

            } /* while not at end of row */
        } /* for each row in the tlist */
    }

    for (currentLoop = 0, i = 0; currentLoop < nc; currentLoop++)
    {
        wrongCt = rightCt = 0;
        while (i < STATE.server->nextLoopTX[currentLoop])
        {
            if (nodes[i].value)
                wrongCt++;
            else
                rightCt++;
            i++;
        }
        /* Overwrite loop endpoint with flag. */
        /* True means that the loop is badly wound */
        STATE.server->nextLoopTX[currentLoop] = (FS_LONG)(wrongCt > rightCt);
    }

    /* Restore flags and pseudo bold */
    STATE.flags = saved_flags;
    STATE.bold_pct = saved_bold_pct;

    return SUCCESS;
}
#endif /* FS_CONTOUR_WINDING_DETECTION */

/****************************************************************/
/* Create a new, emboldened outline from outl. Return the new outline
 * and free the input outline. The embolden amount is STATE.bold_pct
 * of the em length. DO NOT FREE THE INPUT OUTLINE
 */
#ifdef FS_PSEUDO_BOLD
#define CROSS(A,B) (FixMul(A.x, B.y) - FixMul(A.y, B.x))

#if ALDUMP
#define PRVEC(A) FS_PRINTF(("%.2f  %.2f\n", (double)A.x/65536, (double)A.y/65536))
#define PRFIXED(A) FS_PRINTF(("%.4f\n", (double)A/65536))
#endif /* ALDUMP */

FS_OUTLINE *embolden_outline(_DS_
    FS_OUTLINE *outl,    /* Input outline that we have to embolden               */
    FS_LONG *n_types,    /* Number of drawing types in input- return bold number */
    FS_LONG *n_points    /* Number of coordinates in input- return bold coord ct */
)
{
    FS_OUTLINE *bold_outl;           /* The new, bold outline we're making                */
    FIXED_VECTOR bold_vec;           /* vector along which to embolden                    */
    FS_BYTE *type, *b_type;          /* Pointers for stepping through the outline's types */
    FS_FIXED *x, *y, *b_x, *b_y;     /* and coordinates                                   */
    FS_LONG ct_types, ct_points;     /* Counters for the input outline                    */
    FS_LONG bn_types, bn_points;     /* and the bold outline                              */
    FS_LONG ct;
    FS_SHORT dir;        /* Direction path is going: down -1, up +1, horizontal 0 */
    FS_SHORT startdir;   /* Direction at start of path. When we end the path, we will have
                          * to add a "horizontal" segment if the path ends on a different
                          * direction.
                          */
    FS_SHORT beam;       /*  1 if we are making on transitions and
                          * -1 if we are making off transitions
                          */
    FS_SHORT polarity;
    FIXED_VECTOR bold_vec_norm;
    TFNT *tfnt = &STATE.cur_typeset.tfntarray[STATE.cur_font]; /* retrieve current font settings */

#ifdef FS_CONTOUR_WINDING_DETECTION
    FS_SHORT currentLoop = 0;
#endif
#if ALDUMP
    FS_PRINTF(("n_types %d  n_points    %d\n", *n_types, *n_points));
#endif /* ALDUMP */
#ifdef FS_CONTOUR_WINDING_DETECTION
    /* See if we need to check contour winding directions. The emboldening code only
     * works correctly if contour directions are such that "ink is on the right".
     * We don't check for stick fonts because their contours are always in the
     * right direction.
     */
    if (STATE.flags & FLAGS_CHECK_CONTOUR_WINDING_ON &&
            (!(STATE.cur_lfnt->fontflags & FONTFLAG_STIK)))
    {
        check_contour_winding(_PS_ outl);
        if (STATE.error)
            return 0;
    }
#endif


    /* Make the vector along which we embolden. This is (1, 0) sent through
     * the user matrix and then scaled by the bold percent.
     *
     * bold_vec.x = FixMul(sfnt->user_scale[0], STATE.bold_pct);
     * bold_vec.y = FixMul(sfnt->user_scale[2], STATE.bold_pct);
     */
    {
        FS_FIXED fb;

        fb = STATE.lpm * (STATE.bold_pct + tfnt->cfnt->bold_pct);

        bold_vec_norm.x = tfnt->sfnt->user_scale[0];
        bold_vec_norm.y = tfnt->sfnt->user_scale[2];
        fixed_norm(&bold_vec_norm);

        if (STATE.flags & FLAGS_FRACTBOLD_ON)
        {
            /* for CJK grayscale, a fractional offset helps prevent stroke collisions */
            bold_vec.x = FixMul(bold_vec_norm.x, fb);
            bold_vec.y = FixMul(bold_vec_norm.y, fb);
        }
        else
        {
            int b;
            /* usually we want an INTEGER offset .. keeps right sides grid aligned */
            /* if there is no rotation.  */
            b = FS_ROUND(fb);
            if (b == 0)
            {
#ifdef FS_CONTOUR_WINDING_DETECTION
                if ((FS_LONG *)(STATE.server->nextLoopTX))
                {
                    FSS_free(_PS_ (FS_LONG *)(STATE.server->nextLoopTX));
                    STATE.server->nextLoopTX = 0;
                }
#endif
                return outl;
            }

            bold_vec.x = bold_vec_norm.x * b;
            bold_vec.y = bold_vec_norm.y * b;
        }
    }

    /* Set up the bold outline. 3 times the original outline will always be
     * plenty big enough for the bold outline and there are outlines that need 3x.
     */
    bn_types  = 3 * *n_types;
    bn_points = 3 * *n_points;
    bold_outl = setup_outline(_PS_ outl->nc, bn_types, bn_points);

    if (bold_outl == 0)
        return 0;

    polarity = bold_outl->polarity = outl->polarity;

#if ALDUMP
    FS_PRINTF(("bold_vec  "));
    PRVEC(bold_vec);
#endif /* ALDUMP */

    /* Set up the pointers for stepping through the outlines. */
    type   = (FS_BYTE *)( outl->type);
    x      = (FS_FIXED *)(outl->x);
    y      = (FS_FIXED *)(outl->y);
    b_type = (FS_BYTE *)( bold_outl->type);
    b_x    = (FS_FIXED *)(bold_outl->x);
    b_y    = (FS_FIXED *)(bold_outl->y);

    /* Transfer each element from the input outline to the bold outline.
     * Add extra LINETOs when we change direction with respect to the
     * bold_vec.
     */
    ct = ct_types = ct_points =  0;
    while (ct < *n_types)
    {
        /* Start a new loop */
        if ((*type++ & 0x7f) != FS_MOVETO)
        {
#if ALDUMP
            FS_PRINTF(("Bad data; a loop didn't start with a moveto\n"));
#endif /* ALDUMP */
            return 0;
        }
        else
        {
            FS_LONG i;
            FS_FIXED firstdir;
            FIXED_VECTOR A;
#if ALDUMP
            FS_PRINTF(("-----------------------------------\n"));
            FS_PRINTF(("MOVETO  %.2f %.2f\n", (double)(*x) / 65536.0, (double)(*y) / 65536.0));
#endif /* ALDUMP */
            ct++;
            *b_type++ = FS_MOVETO;
            *b_x++ = *x++;
            *b_y++ = *y++;
            ct_types++;
            ct_points++;
            /* Set 1st direction. Move forward along curve and use 1st
              * non horizontal as 1st direction.
             */
#if ALDUMP
            FS_PRINTF(("    Finding first direction\n"));
#endif /* ALDUMP */
            for (i = 0, firstdir = 0; i < *n_types && type[i] != FS_MOVETO && firstdir == 0; i++)
            {
                A.x = x[i] - x[i - 1];
                A.y = y[i] - y[i - 1];
                firstdir = CROSS(bold_vec, A);
#if ALDUMP
                FS_PRINTF(("        Next vec: "));
                PRVEC(A);
                FS_PRINTF(("        dir vector  %d\n\n", firstdir));
#endif /* ALDUMP */
            }
            if (firstdir > 0)
                dir = 1;
            else if (firstdir < 0)
                dir = -1;
            else /* We have a collapsed loop parallel to the bold vec. */
            {
                dir = 1;  /* This will cause the "loop" to be copied as is */
#if ALDUMP
                FS_PRINTF(("WARNING: a collapsed loop\n"));
#endif /* ALDUMP */
            }
            startdir = dir;

            if (polarity != 0)
                beam = polarity * dir; /* assumes ink on the right for polarity == 1 */
            else
                beam = dir;

#ifdef FS_CONTOUR_WINDING_DETECTION
            if ((STATE.server->nextLoopTX) && (STATE.server->nextLoopTX[currentLoop++]))
                beam = -beam;
#endif
            if (beam == -1)
            {
                b_x[-1] += bold_vec.x; /* all "off" pts get shifted */
                b_y[-1] += bold_vec.y;
            }
#if ALDUMP
            FS_PRINTF(("    moveto  %.2f  %.2f\n", (double)b_x[-1] / 65536.0, (double)b_y[-1] / 65536.0));
#endif /* ALDUMP */
        }
        while (ct < *n_types && *type != FS_MOVETO) /* Process the rest of the loop */
        {
            ct++;
            switch (*type++)
            {
            case FS_LINETO:
            {
                FS_FIXED newdir;
                FIXED_VECTOR A;
#if ALDUMP
                FS_PRINTF(("LINETO  %.2f %.2f\n", (double)(*x) / 65536.0, (double)(*y) / 65536.0));
#endif /* ALDUMP */

                /* The sign of the cross product determines whether we're going
                * up, down, or horizontal with respect to the bold vector. If it
                * is different in sign than dir, then we've changed direction.
                */
                A.x = x[0] - x[-1];
                A.y = y[0] - y[-1];
                newdir = dir * CROSS(bold_vec, A);
#if ALDUMP
                FS_PRINTF(("newdir %d\n", newdir));
#endif /* ALDUMP */
                if (newdir < 0)
                {
                    /* changed direction; add extra line */
#if ALDUMP
                    FS_PRINTF(("    Changed direction\n"));
#endif /* ALDUMP */
                    *b_type++ = FS_LINETO;
                    *b_x = b_x[-1] + beam * bold_vec.x;
                    *b_y = b_y[-1] + beam * bold_vec.y;
                    b_x++;
                    b_y++;
                    beam = -beam;
                    dir = -dir;
#if ALDUMP
                    FS_PRINTF(("dir %d\n", dir));
                    FS_PRINTF(("Bold pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    ct_types++;
                    ct_points++;
                }
                /* Next copy the original lineto */
                *b_type++ = FS_LINETO;
                *b_x = b_x[-1] + A.x;
                *b_y = b_y[-1] + A.y;
                b_x++;
                b_y++;
                x++;
                y++;
#if ALDUMP
                FS_PRINTF(("    linteto  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                ct_types++;
                ct_points++;
                break;
            }
            case FS_QUADTO:
            {
                FS_FIXED newdir1, newdir2;
                FIXED_VECTOR A, B;
                FIXED_VECTOR nA, nB;
#if ALDUMP
                FS_PRINTF(("QUADTO  %.2f %.2f      %.2f %.2f\n", (double)(*x) / 65536.0, (double)(*y) / 65536.0, (double)(x[1]) / 65536.0, (double)(y[1]) / 65536.0));
#endif /* ALDUMP */

                nA.x = A.x = x[0] - x[-1];
                nA.y = A.y = y[0] - y[-1];
                nB.x = B.x = x[1] - x[0];
                nB.y = B.y = y[1] - y[0];
                fixed_norm(&nA);
                fixed_norm(&nB);
                newdir1 = dir * CROSS(bold_vec_norm, nA);
                newdir2 = dir * CROSS(bold_vec_norm, nB);
                if (ABS(newdir1) < 16384)
                    newdir1 = 0;
                if (ABS(newdir2) < 16384)
                    newdir2 = 0;
#if ALDUMP
                FS_PRINTF(("    newdir1 %ld    newdir2  %ld\n", newdir1, newdir2));
#endif /* ALDUMP */
                if ((newdir1 < 0) || (newdir1 == 0 && newdir2 < 0))
                {
#if ALDUMP
                    FS_PRINTF(("    Changed direction\n"));
#endif /* ALDUMP */
                    /* changed direction; add extra line */
                    *b_type++ = FS_LINETO;
                    *b_x = b_x[-1] + beam * bold_vec.x;
                    *b_y = b_y[-1] + beam * bold_vec.y;
                    b_x++;
                    b_y++;
                    dir = -dir;
                    beam = -beam;
#if ALDUMP
                    FS_PRINTF(("dir %d\n", dir));
                    FS_PRINTF(("Bold pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    ct_types++;
                    ct_points++;
                }
                if ((newdir1 > 0 && newdir2 < 0) || (newdir1 < 0 && newdir2 > 0))
                {
                    /* need to break the quad */
                    FIXED_VECTOR K;
                    FS_FIXED t0, t0sq;
#if ALDUMP
                    FS_PRINTF(("\n\nBreaking quad\n"));
#endif /* ALDUMP */

                    K.x = B.x - A.x;
                    K.y = B.y - A.y;
#if ALDUMP
                    FS_PRINTF(("A: "));
                    PRVEC(A);
                    FS_PRINTF(("B: "));
                    PRVEC(B);
                    FS_PRINTF(("K: "));
                    PRVEC(K);
#endif /* ALDUMP */

                    t0 = -FixDiv(CROSS(bold_vec, A), CROSS(bold_vec, K));
#if ALDUMP
                    FS_PRINTF(("t0  "));
                    PRFIXED(t0);
#endif /* ALDUMP */
                    if (!(0 <= t0 && t0 <= FIXED_ONE))
                    {
#if ALDUMP
                        FS_PRINTF(("t0 is out of range, this can't happen\n"));
#endif /* ALDUMP */
                    }

                    /* 1st quad */
                    *b_type++ = FS_QUADTO;
#if ALDUMP
                    FS_PRINTF(("start pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    /* control point */
                    *b_x = b_x[-1] + FixMul(t0, A.x);
                    *b_y = b_y[-1] + FixMul(t0, A.y);
                    b_x++;
                    b_y++;
#if ALDUMP
                    FS_PRINTF(("    control pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    /* end point */
                    t0sq = FixMul(t0, t0);
                    *b_x = b_x[-2] + 2 * FixMul(A.x, t0) + FixMul(K.x, t0sq);
                    *b_y = b_y[-2] + 2 * FixMul(A.y, t0) + FixMul(K.y, t0sq);
                    b_x++;
                    b_y++;
#if ALDUMP
                    FS_PRINTF(("    end pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */

                    /* bold segment */
                    *b_type++ = FS_LINETO;
                    *b_x = b_x[-1] + beam * bold_vec.x;
                    *b_y = b_y[-1] + beam * bold_vec.y;
                    b_x++;
                    b_y++;
#if ALDUMP
                    FS_PRINTF(("    bold pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    beam = -beam;
                    dir = -dir;
#if ALDUMP
                    FS_PRINTF(("dir %d\n", dir));
#endif /* ALDUMP */

                    /* 2nd quad */
                    *b_type++ = FS_QUADTO;
                    b_x[1] = x[1];   /* 1st the end point */
                    b_y[1] = y[1];
                    if (dir < 0)
                    {
                        b_x[1] += bold_vec.x;
                        b_y[1] += bold_vec.y;
                    }
                    b_x[0] = b_x[1] + FixMul(t0 - FIXED_ONE, B.x); /* next the control point */
                    b_y[0] = b_y[1] + FixMul(t0 - FIXED_ONE, B.y);
#if ALDUMP
                    FS_PRINTF(("    cntrl pt  %.2f  %.2f\n", (double)b_x[0] / 65536, (double)b_y[0] / 65536));
                    FS_PRINTF(("    end pt  %.2f  %.2f\n", (double)b_x[1] / 65536, (double)b_y[1] / 65536));
#endif /* ALDUMP */

                    /* update ptrs and ctrs */
                    x   += 2;
                    y   += 2;
                    b_x += 2;
                    b_y += 2;
                    ct_types  += 3;
                    ct_points += 5;
                }
                else
                {
                    *b_type++ = FS_QUADTO;
                    *b_x = b_x[-1] + A.x;  /* Use relative to avoid having to check */
                    *b_y = b_y[-1] + A.y;  /* dir and adding bold_vec.              */
                    b_x++;
                    b_y++;
                    *b_x = b_x[-1] + B.x;
                    *b_y = b_y[-1] + B.y;
                    b_x++;
                    b_y++;
#if ALDUMP
                    FS_PRINTF(("    control pt  %.2f  %.2f\n", (double)b_x[-2] / 65536, (double)b_y[-2] / 65536));
                    FS_PRINTF(("    end pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */

                    x += 2;
                    y += 2;
                    ct_types++;
                    ct_points += 2;
                }
                break;
            } /* case FS_QUADTO */
            case FS_CUBETO:
            {
                FS_FIXED newdir1, newdir2;
                FIXED_VECTOR A, B, C;
                FIXED_VECTOR nA, nB;
#if ALDUMP
                FS_PRINTF(("QUADTO  %.2f %.2f      %.2f %.2f\n", (double)(*x) / 65536.0, (double)(*y) / 65536.0, (double)(x[1]) / 65536.0, (double)(y[1]) / 65536.0));
#endif /* ALDUMP */

                nA.x = A.x = x[0] - x[-1];
                nA.y = A.y = y[0] - y[-1];
                nB.x = B.x = x[2] - x[1];
                nB.y = B.y = y[2] - y[1];
                C.x = x[1] - x[0];
                C.y = y[1] - y[0];
                fixed_norm(&nA);
                fixed_norm(&nB);
                newdir1 = dir * CROSS(bold_vec_norm, nA);
                newdir2 = dir * CROSS(bold_vec_norm, nB);
                if (ABS(newdir1) < 16384)
                    newdir1 = 0;
                if (ABS(newdir2) < 16384)
                    newdir2 = 0;
#if ALDUMP
                FS_PRINTF(("    newdir1 %ld    newdir2  %ld\n", newdir1, newdir2));
#endif /* ALDUMP */
                if ((newdir1 < 0) || (newdir1 == 0 && newdir2 < 0))
                {
#if ALDUMP
                    FS_PRINTF(("    Changed direction\n"));
#endif /* ALDUMP */
                    /* changed direction; add extra line */
                    *b_type++ = FS_LINETO;
                    *b_x = b_x[-1] + beam * bold_vec.x;
                    *b_y = b_y[-1] + beam * bold_vec.y;
                    b_x++;
                    b_y++;
                    dir = -dir;
                    beam = -beam;
#if ALDUMP
                    FS_PRINTF(("dir %d\n", dir));
                    FS_PRINTF(("Bold pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    ct_types++;
                    ct_points++;
                }
                if ((newdir1 > 0 && newdir2 < 0) || (newdir1 < 0 && newdir2 > 0))
                {


                    *b_type++ = FS_CUBETO;
#if ALDUMP
                    FS_PRINTF(("start pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
                    /* control point */
                    *b_x = b_x[-1] + A.x;
                    *b_y = b_y[-1] + A.y;

                    b_x[2] = x[2];   /* 1st the end point */
                    b_y[2] = y[2];
                    if (dir < 0)
                    {
                        b_x[2] += bold_vec.x;
                        b_y[2] += bold_vec.y;
                    }


                    b_x[1] = b_x[2] - B.x;  /* next the control point */
                    b_y[1] = b_y[2] - B.y;
#if ALDUMP
                    FS_PRINTF(("    cntrl pt  %.2f  %.2f\n", (double)b_x[0] / 65536, (double)b_y[0] / 65536));
                    FS_PRINTF(("    end pt  %.2f  %.2f\n", (double)b_x[1] / 65536, (double)b_y[1] / 65536));
#endif /* ALDUMP */
                    beam = -beam;
                    dir = -dir;
                    /* update ptrs and ctrs */
                    x   += 3;
                    y   += 3;
                    b_x += 3;
                    b_y += 3;
                    ct_types  += 1;
                    ct_points += 3;
                }
                else
                {
                    *b_type++ = FS_CUBETO;
                    *b_x = b_x[-1] + A.x;  /* Use relative to avoid having to check */
                    *b_y = b_y[-1] + A.y;  /* dir and adding bold_vec.              */
                    b_x++;
                    b_y++;
                    *b_x = b_x[-1] + C.x;
                    *b_y = b_y[-1] + C.y;
                    b_x++;
                    b_y++;
                    *b_x = b_x[-1] + B.x;
                    *b_y = b_y[-1] + B.y;
                    b_x++;
                    b_y++;

#if ALDUMP
                    FS_PRINTF(("    control pt  %.2f  %.2f\n", (double)b_x[-2] / 65536, (double)b_y[-2] / 65536));
                    FS_PRINTF(("    end pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */

                    x += 3;
                    y += 3;
                    ct_types++;
                    ct_points += 3;
                }
                break;
            } /* case FS_CUBETO */
            default:
                break;
            }  /* switch */
        } /* while */
        /* End of loop */

        /* If start and ending dir are different, we need to add a bold lineto
         * so that we connect up to the start point.
        */
        if (dir != startdir)
        {
            /* changed direction; add extra line */
#if ALDUMP
            FS_PRINTF(("    Changed direction\n"));
#endif /* ALDUMP */
            *b_type++ = FS_LINETO;
            *b_x = b_x[-1] + beam * bold_vec.x;
            *b_y = b_y[-1] + beam * bold_vec.y;
            b_x++;
            b_y++;
#if ALDUMP
            FS_PRINTF(("Bold pt  %.2f  %.2f\n", (double)b_x[-1] / 65536, (double)b_y[-1] / 65536));
#endif /* ALDUMP */
            ct_types++;
            ct_points++;
        }
    }
    bold_outl->type[0] |= OUTLINE_CHAR;
#ifdef FS_CONTOUR_WINDING_DETECTION
    if ((FS_LONG *)(STATE.server->nextLoopTX))
    {
        FSS_free(_PS_ (FS_LONG *)(STATE.server->nextLoopTX));
        STATE.server->nextLoopTX = 0;
    }
#endif


    /*** set the advance width ***/
    {
#ifdef FS_MONOSPACE_PSEUDOBOLD_ADVANCE
        TTF *ttf = (TTF *)STATE.cur_lfnt->fnt;
        TTF_OS2 *pos2;

        pos2 = (TTF_OS2 *)ttf->os2;
        if (pos2 && pos2->panose[3] == 9)
        {
            /* monospaced font, do not change the advance width */
            bold_outl->dx = outl->dx;
            bold_outl->dy = outl->dy;
            bold_outl->i_dx = outl->i_dx;
            bold_outl->i_dy = outl->i_dy;

            if (STATE.flags & FLAGS_HINTS_OFF)
            {
                /* adjust outline to keep stems in same effective position */
                FS_FIXED xshift = bold_vec.x>>1;
                FS_LONG i;

                for (i=0; i<bn_points; i++)
                {
                    bold_outl->x[i] -= xshift;
                }
                bold_outl->lo_x -= xshift;
                bold_outl->hi_x -= xshift;
            }

        }
        else
#endif
        {
            if(outl->dx || outl->dy)
            {
                bold_outl->dx = outl->dx + bold_vec.x;
                bold_outl->dy = outl->dy + bold_vec.y;
                if (outl->i_dx)
                    bold_outl->i_dx = outl->i_dx + FS_ROUND(bold_vec.x);
                if (outl->i_dy)
                    bold_outl->i_dy = outl->i_dy + FS_ROUND(bold_vec.y);
            }
            else
            {
                /* handle zero-width diacritics */
                bold_outl->dx = outl->dx;
                bold_outl->dy = outl->dy;
                if (outl->i_dx)
                    bold_outl->i_dx = outl->i_dx;
                if (outl->i_dy)
                    bold_outl->i_dy = outl->i_dy;
            }
        }
    }
    
    /*** set the bbox ***/
    set_bbox(ct_points, bold_outl);
    /* Return the new type and pt counts. */
    *n_types = ct_types;
    *n_points = ct_points;
#if ALDUMP
    FS_PRINTF(("n_types %d  n_points    %d\n", *n_types, *n_points));
#endif /* ALDUMP */

    /* Scrunch down the probably way-too-big bold outline
     * and realloc to save cache space.
     */
    if (ct_types < bn_types || ct_points < bn_points)
    {
        FS_OUTLINE *small_outl;
        FS_BYTE *stype, *btype;
        FS_FIXED *sx, *bx;
        FS_FIXED *sy, *by;
        small_outl = setup_outline(_PS_ outl->nc, ct_types, ct_points);

        if (!small_outl)
        {
            /* it's larger than necessary, but still OK */
            return bold_outl;
        }
        stype = (FS_BYTE *)(small_outl->type);
        btype = (FS_BYTE *)(bold_outl->type);
        SYS_MEMCPY(stype, btype, ct_types);

        sx = (FS_FIXED *)(small_outl->x);
        bx = (FS_FIXED *)(bold_outl->x);
        SYS_MEMCPY(sx, bx, 4 * ct_points);

        sy = (FS_FIXED *)(small_outl->y);
        by = (FS_FIXED *)(bold_outl->y);
        SYS_MEMCPY(sy, by, 4 * ct_points);

        /* gotta copy bbox and escapements too */
        small_outl->lo_x = bold_outl->lo_x;
        small_outl->hi_x = bold_outl->hi_x;
        small_outl->lo_y = bold_outl->lo_y;
        small_outl->hi_y = bold_outl->hi_y;
        small_outl->dx = bold_outl->dx;
        small_outl->dy = bold_outl->dy;
        small_outl->i_dx = bold_outl->i_dx;
        small_outl->i_dy = bold_outl->i_dy;
        small_outl->polarity = bold_outl->polarity;
        FSS_free_char(_PS_ bold_outl);
        return small_outl;
    }
    else
        return bold_outl;
}

#endif /* PSEUDO-BOLD */
/****************************************************************/
#endif /* FS_RENDER */
